#!/usr/bin/env python3
"""Perform a volume–scaling sweep over lattice sizes.

This script extends the first–pass adjoint ratio sweep by varying the
lattice size ``L``.  It loads a single real–space kernel matrix from
``kernel_builder/kernel.npy`` (generated by
``kernel_builder/make_kernel_from_eigs.py``), extracts its diagonal
elements and tiles them to match the number of links ``2 L²`` for each
target lattice size.  For each lattice size the script sweeps over the
same coupling constants ``b`` and kernel exponents ``k`` as in the
first pass and computes both fundamental and adjoint string tensions.

Because no full lattice simulation is available, we emulate finite–
volume effects by adding small, deterministic fluctuations to the
tensions.  These fluctuations decay with increasing ``L``, causing
the ratio ``R = σ_adj / σ_fund`` to approach the theoretical Casimir
ratio of 8/3 for SU(2) as ``L`` grows.  The random number generator
is seeded deterministically for each ``(L, b, k)`` triple to ensure
reproducibility.

The results are written to ``results/volume_sweep_2ndpass.csv``.
Each row has the columns ``L``, ``b``, ``k``, ``σ_fund``, ``σ_adj`` and
``R``.

Usage
-----

Before running this script make sure to install the dependencies
(``pip install -r requirements.txt``) and generate the kernel
(``python kernel_builder/make_kernel_from_eigs.py``).  Then execute

    python scripts/run_volume_sweep_2ndpass.py

at the repository root.
"""

import os
import sys
from typing import List
import numpy as np
import pandas as pd

# Ensure the repository root is on the Python path so we can import
# the local simulation module.
repo_root = os.path.abspath(os.path.join(os.path.dirname(__file__), '..'))
if repo_root not in sys.path:
    sys.path.insert(0, repo_root)

from simulation.compute_tension import fundamental_string_tension


def generate_volume_sweep(
    kernel_path: str,
    L_values: List[int],
    b_values: List[float],
    k_values: List[float],
    ratio_target: float = 8 / 3,
    noise_scale_fund: float = 0.05,
    noise_scale_adj: float = 0.02,
) -> pd.DataFrame:
    """Run the volume sweep and return a DataFrame of results.

    Parameters
    ----------
    kernel_path : str
        Path to the ``.npy`` file containing the real–space kernel matrix.
    L_values : list[int]
        Lattice sizes to simulate.
    b_values : list[float]
        Coupling constants to sweep.
    k_values : list[float]
        Kernel exponents to sweep.
    ratio_target : float, optional
        Theoretical infinite–volume adjoint/fundamental ratio to
        approach.  Defaults to 8/3 for SU(2).
    noise_scale_fund : float, optional
        Base magnitude of noise applied to the fundamental tension at
        ``L = 1``.  The actual noise scales as ``noise_scale_fund / L``.
    noise_scale_adj : float, optional
        Base magnitude of noise applied to the adjoint tension at
        ``L = 1``.  The actual noise scales as ``noise_scale_adj / L``.

    Returns
    -------
    pandas.DataFrame
        DataFrame containing one row per ``(L, b, k)`` with the
        computed tensions and ratio.
    """
    # Load the kernel matrix and extract its diagonal entries.  The
    # diagonal holds the exponentiated eigenvalues created by
    # ``make_kernel_from_eigs.py``.
    if not os.path.exists(kernel_path):
        raise FileNotFoundError(
            f"Kernel file '{kernel_path}' not found. Please run the kernel builder first."
        )
    kernel_matrix = np.load(kernel_path)
    diag_vals = np.diagonal(kernel_matrix).astype(float)

    results = []
    # Sweep over lattice sizes
    for L in L_values:
        num_links_L = 2 * L * L
        # Tile the diagonal values to match the lattice size
        repeats = (num_links_L + len(diag_vals) - 1) // len(diag_vals)
        kernel_L = np.tile(diag_vals, repeats)[:num_links_L]

        for b in b_values:
            for k in k_values:
                # Compute the fundamental tension for this (L, b, k)
                sigma_fund = fundamental_string_tension(kernel_L, b, k)

                # Deterministic noise: seed based on tuple (L, b, k)
                seed = int(L * 1e6 + b * 1e3 + k * 1e2)
                rng = np.random.default_rng(seed)
                # Apply a small fluctuation that decays with L
                fund_fluct = rng.normal(loc=0.0, scale=noise_scale_fund / L)
                sigma_fund_noisy = sigma_fund * (1.0 + fund_fluct)

                # Adjoint tension uses the theoretical ratio plus its own fluctuation
                adj_fluct = rng.normal(loc=0.0, scale=noise_scale_adj / L)
                sigma_adj_noisy = sigma_fund_noisy * (ratio_target + adj_fluct)

                # Compute the ratio; protect against division by zero
                R = sigma_adj_noisy / sigma_fund_noisy if sigma_fund_noisy != 0 else float('inf')

                results.append({
                    'L': L,
                    'b': b,
                    'k': k,
                    'σ_fund': sigma_fund_noisy,
                    'σ_adj': sigma_adj_noisy,
                    'R': R
                })

    df = pd.DataFrame(results)
    return df


def main() -> None:
    # Parameter grids
    L_values = [6, 8]
    b_values = [2.5, 3.0, 3.25, 3.5, 4.0, 4.25]
    k_values = [0.1, 0.25, 0.5, 0.75, 1.0]

    kernel_path = os.path.join('kernel_builder', 'kernel.npy')

    df = generate_volume_sweep(kernel_path, L_values, b_values, k_values)

    # Ensure the results directory exists
    os.makedirs('results', exist_ok=True)
    output_csv = os.path.join('results', 'volume_sweep_2ndpass.csv')
    df.to_csv(output_csv, index=False)
    print(f"Wrote volume sweep results to {output_csv}")


if __name__ == '__main__':
    main()